const express = require('express');
const multer = require('multer');
const { spawn } = require('child_process');
const path = require('path');
const fs = require('fs');
const cors = require('cors');
const { v4: uuidv4 } = require('uuid');

const app = express();
app.use(cors());
app.use(express.json());

const storage = multer.diskStorage({
    destination: './uploads/',
    filename: (req, file, cb) => cb(null, uuidv4() + path.extname(file.originalname))
});
const upload = multer({ storage, limits: { fileSize: 500 * 1024 * 1024 } });

const jobs = new Map();

// Auto-cleanup every 15 minutes
setInterval(() => {
    const now = Date.now();
    for (const [jobId, job] of jobs.entries()) {
        if (now - job.createdAt > 30 * 60 * 1000) { // 30 min
            try {
                if (job.inputPath && fs.existsSync(job.inputPath)) fs.unlinkSync(job.inputPath);
                if (job.outputPath && fs.existsSync(job.outputPath)) fs.unlinkSync(job.outputPath);
                const trf = path.join(__dirname, 'temp', `${jobId}.trf`);
                if (fs.existsSync(trf)) fs.unlinkSync(trf);
            } catch (e) {
                console.error(`Cleanup error for ${jobId}:`, e.message);
            }
            jobs.delete(jobId);
        }
    }
}, 15 * 60 * 1000);

function runFFmpeg(args) {
    return new Promise((resolve, reject) => {
        console.log('FFmpeg:', args.join(' '));
        const proc = spawn('ffmpeg', args);
        let stderr = '';
        proc.stderr.on('data', d => stderr += d.toString());
        proc.on('close', code => {
            if (code === 0) resolve();
            else {
                console.error('FFmpeg error:', stderr);
                reject(new Error(`FFmpeg exited with ${code}`));
            }
        });
    });
}

async function stabilizeVideo(jobId, inputPath, outputPath, settings) {
    console.log(`[${jobId}] START ${path.basename(inputPath)}`);

    try {
        const job = jobs.get(jobId);
        if (!job) throw new Error('Job not found');

        // Build trim args
        const trimArgs = [];
        if (settings.trimEnd > 0 && settings.trimEnd > settings.trimStart) {
            if (settings.trimStart > 0) trimArgs.push('-ss', settings.trimStart.toString());
            const duration = settings.trimEnd - settings.trimStart;
            if (duration > 0) trimArgs.push('-t', duration.toString());
        }

        console.log(`[${jobId}] Trim: ${settings.trimStart}s → ${settings.trimEnd}s`);

        job.status = 'stabilizing';
        job.progress = 30;

        // Build filter chain using DESHAKE (built into FFmpeg - no vidstab needed!)
        const filters = [];

        // Map shakiness (1-10) to deshake rx/ry (16-80 pixels)
        const shakiness = settings.shakiness || 5;
        const rx = Math.round(8 + (shakiness * 7.2)); // 16-80 pixels
        const ry = rx;
        
        // Map smoothing (1-100) to blocksize and search
        const smoothing = settings.smoothing || 30;
        const blocksize = smoothing < 30 ? 16 : 8; // Larger blocks = smoother
        const contrast = smoothing < 30 ? 100 : 150;
        
        // Build deshake filter
        let deshakeFilter = `deshake=x=-1:y=-1:w=-1:h=-1:rx=${rx}:ry=${ry}:edge=mirror:blocksize=${blocksize}:contrast=${contrast}:search=1`;
        filters.push(deshakeFilter);

        // Apply zoom/crop if specified
        if (settings.zoom && settings.zoom > 0) {
            const scale = 1 + (settings.zoom / 100);
            filters.push(`scale=iw*${scale}:ih*${scale}`);
            filters.push('crop=iw/1.02:ih/1.02'); // Slight crop to remove edges
        }

        // Jello correction (additional deshake pass with different params)
        if (settings.jelloIntensity > 0) {
            const jelloRx = Math.min(settings.jelloIntensity * 4, 32);
            filters.push(`deshake=rx=${jelloRx}:ry=${jelloRx}:edge=mirror:blocksize=8`);
        }

        // Rolling shutter fix
        if (settings.rollingShutter) {
            filters.push('deshake=rx=32:ry=32:edge=mirror');
        }

        // Denoise
        if (settings.denoise > 0) {
            const s = Math.min(settings.denoise * 1.5, 8);
            filters.push(`hqdn3d=${s}:${s}:${s*.5}:${s*.5}`);
        }

        // Unsharp mask
        if (settings.unsharp > 0) {
            filters.push(`unsharp=5:5:${settings.unsharp}:5:5:0.0`);
        }

        console.log(`[${jobId}] Filters: ${filters.join(' → ')}`);

        job.progress = 50;

        await runFFmpeg([
            ...trimArgs,
            '-i', inputPath,
            '-vf', filters.join(','),
            '-c:v', 'libx264',
            '-preset', 'medium',
            '-crf', '18',
            '-c:a', 'copy',
            outputPath
        ]);

        job.status = 'completed';
        job.progress = 100;
        console.log(`[${jobId}] ✓ Stabilized`);

    } catch (err) {
        console.error(`[${jobId}] ERR:`, err);
        const job = jobs.get(jobId);
        if (job) {
            job.status = 'failed';
            job.error = err.message;
        }
        throw err;
    }
}

// API Routes
app.get('/api/health', (req, res) => {
    res.json({ status: 'ok', message: 'Server is running' });
});

app.post('/api/stabilize', upload.single('video'), async (req, res) => {
    if (!req.file) return res.status(400).json({ error: 'No video uploaded' });

    const jobId = uuidv4();
    const inputPath = req.file.path;
    const outputPath = path.join(__dirname, 'outputs', `${jobId}_stabilized.mp4`);

    const settings = {
        shakiness: parseInt(req.body.shakiness) || 5,
        smoothing: parseInt(req.body.smoothing) || 30,
        zoom: parseInt(req.body.zoom) || 0,
        rollingShutter: req.body.rollingShutter === '1',
        jelloIntensity: parseInt(req.body.jelloIntensity) || 0,
        unsharp: parseFloat(req.body.unsharp) || 0,
        denoise: parseInt(req.body.denoise) || 0,
        interpolation: req.body.interpolation || 'bicubic',
        preset: req.body.preset || 'custom',
        trimStart: parseFloat(req.body.trimStart) || 0,
        trimEnd: parseFloat(req.body.trimEnd) || 0,
        duration: parseFloat(req.body.duration) || 0,
        email: req.body.email || null
    };

    jobs.set(jobId, {
        jobId,
        inputPath,
        outputPath,
        status: 'queued',
        progress: 0,
        createdAt: Date.now(),
        settings
    });

    res.json({ jobId, message: 'Stabilization started' });

    stabilizeVideo(jobId, inputPath, outputPath, settings).catch(err => {
        console.error('Stabilization error:', err);
    });
});

app.get('/api/status/:jobId', (req, res) => {
    const job = jobs.get(req.params.jobId);
    if (!job) return res.status(404).json({ error: 'Job not found' });
    
    res.json({
        jobId: job.jobId,
        status: job.status,
        progress: job.progress,
        error: job.error || null
    });
});

app.get('/api/download/:jobId', (req, res) => {
    const job = jobs.get(req.params.jobId);
    if (!job) return res.status(404).json({ error: 'Job not found' });
    if (job.status !== 'completed') return res.status(400).json({ error: 'Video not ready' });
    if (!fs.existsSync(job.outputPath)) return res.status(404).json({ error: 'File not found' });

    res.download(job.outputPath, `stabilized_${job.jobId}.mp4`, err => {
        if (err) console.error('Download error:', err);
    });
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
    console.log(`✓ Running on ${PORT}`);
});
