#!/bin/bash

# Video Stabilizer - Mac Setup Script (Simplified - No vidstab needed!)
# This script installs all dependencies needed to run the local video stabilizer

echo "🎬 Video Stabilizer - Mac Setup"
echo "================================"
echo ""
echo "✨ This version uses FFmpeg's built-in deshake filter"
echo "   No special compilation needed - works with any FFmpeg!"
echo ""

# Get the directory where this script is located
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
cd "$SCRIPT_DIR"

echo "📁 Working directory: $SCRIPT_DIR"
echo ""

# Function to check if a command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# 1. Check for Homebrew
echo "🍺 Checking for Homebrew..."
if ! command_exists brew; then
    echo "❌ Homebrew not found. Installing..."
    /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
    
    # Add Homebrew to PATH for this session
    if [[ $(uname -m) == 'arm64' ]]; then
        eval "$(/opt/homebrew/bin/brew shellenv)"
    else
        eval "$(/usr/local/bin/brew shellenv)"
    fi
else
    echo "✅ Homebrew is already installed"
fi
echo ""

# 2. Check for Node.js
echo "📦 Checking for Node.js..."
if ! command_exists node; then
    echo "❌ Node.js not found. Installing via Homebrew..."
    brew install node
    
    if [ $? -ne 0 ]; then
        echo "❌ Failed to install Node.js!"
        echo "Please install manually from: https://nodejs.org"
        read -p "Press Enter to close..."
        exit 1
    fi
else
    echo "✅ Node.js is already installed ($(node --version))"
fi
echo ""

# 3. Check for FFmpeg (ANY version works now!)
echo "🎥 Checking for FFmpeg..."
if ! command_exists ffmpeg; then
    echo "❌ FFmpeg not found. Installing via Homebrew..."
    brew install ffmpeg
    
    if [ $? -ne 0 ]; then
        echo "❌ Failed to install FFmpeg!"
        echo "Please install manually: brew install ffmpeg"
        read -p "Press Enter to close..."
        exit 1
    fi
else
    echo "✅ FFmpeg is already installed"
fi

# Verify FFmpeg has deshake filter (it should - it's built-in!)
if ffmpeg -filters 2>/dev/null | grep -q "deshake"; then
    echo "✅ deshake filter found! (Stabilization will work)"
else
    echo "⚠️  deshake filter not found (unusual - might not work)"
fi
echo ""

# 4. Install npm dependencies
echo "📚 Installing npm packages..."
if [ -f "package.json" ]; then
    echo ""
    echo "⚙️  Running: sudo npm install --unsafe-perm"
    echo "You will need to enter your Mac password."
    echo ""
    
    sudo npm install --unsafe-perm
    
    if [ $? -ne 0 ]; then
        echo ""
        echo "❌ npm install failed!"
        echo ""
        echo "Please try manually:"
        echo "1. Open Terminal"
        echo "2. Run: cd ~/Downloads/stabilizer-package-v4"
        echo "3. Run: sudo npm install --unsafe-perm"
        echo ""
        read -p "Press Enter to close..."
        exit 1
    fi
    
    echo ""
    echo "✅ npm packages installed successfully!"
else
    echo "❌ package.json not found in $SCRIPT_DIR"
    echo "Make sure you've extracted all files from the stabilizer package!"
    read -p "Press Enter to close..."
    exit 1
fi
echo ""

# 5. Create necessary directories
echo "📂 Creating required directories..."
mkdir -p uploads outputs temp
echo "✅ Directories created"
echo ""

# 6. Final verification
echo "🔍 Final verification..."
ALL_GOOD=true

if ! command_exists node; then
    echo "❌ Node.js not found"
    ALL_GOOD=false
fi

if ! command_exists ffmpeg; then
    echo "❌ FFmpeg not found"
    ALL_GOOD=false
fi

if [ ! -d "node_modules/axios" ] || [ ! -d "node_modules/express" ]; then
    echo "⚠️  Some npm packages may be missing"
    ALL_GOOD=false
fi

if [ "$ALL_GOOD" = true ]; then
    echo "✅ All components verified!"
else
    echo "⚠️  Some components may have issues. Check messages above."
fi
echo ""

# Done!
echo "================================"
echo "✅ Setup Complete!"
echo "================================"
echo ""
echo "📋 Next Steps:"
echo "1. Close this window"
echo "2. Right-click 'start.command' and select 'Open'"
echo "3. Go to your browser and use the stabilizer!"
echo ""
echo "💡 Tip: You only need to run setup.command once."
echo "   Just run start.command each time you want to use the stabilizer."
echo ""
echo "🎯 This version uses deshake filter (no vidstab needed!)"
echo ""

# Keep window open so user can read the message
read -p "Press Enter to close this window..."
