#!/bin/bash

# Video Stabilizer - Server Startup Script
# Double-click this file each time you want to use the stabilizer

echo "🎬 Starting Video Stabilizer Server..."
echo "======================================"
echo ""

# Get the directory where this script is located
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
cd "$SCRIPT_DIR"

echo "📁 Working directory: $SCRIPT_DIR"
echo ""

# Add Homebrew paths to environment so Node can find FFmpeg
if [[ $(uname -m) == 'arm64' ]]; then
    # Apple Silicon Mac
    export PATH="/opt/homebrew/bin:$PATH"
    echo "🍺 Added Apple Silicon Homebrew path"
else
    # Intel Mac
    export PATH="/usr/local/bin:$PATH"
    echo "🍺 Added Intel Homebrew path"
fi

# Check if node is available
if ! command -v node >/dev/null 2>&1; then
    echo "❌ Node.js not found!"
    echo ""
    echo "Please run setup.command first to install dependencies."
    echo ""
    read -p "Press Enter to close..."
    exit 1
fi

# Check if FFmpeg is available
if ! command -v ffmpeg >/dev/null 2>&1; then
    echo "❌ FFmpeg not found!"
    echo ""
    echo "Please run setup.command first to install FFmpeg."
    echo ""
    read -p "Press Enter to close..."
    exit 1
fi

echo "✅ FFmpeg found: $(ffmpeg -version | head -n1)"

# Check if package.json exists
if [ ! -f "package.json" ]; then
    echo "❌ package.json not found!"
    echo ""
    echo "Make sure you're running this script from the stabilizer folder."
    echo ""
    read -p "Press Enter to close..."
    exit 1
fi

# Check if node_modules exists
if [ ! -d "node_modules" ]; then
    echo "⚠️  Dependencies not installed. Running npm install..."
    echo ""
    sudo npm install --unsafe-perm
    
    if [ $? -ne 0 ]; then
        echo ""
        echo "❌ npm install failed!"
        echo ""
        read -p "Press Enter to close..."
        exit 1
    fi
fi

# Verify critical modules
echo "🔍 Verifying dependencies..."
if [ ! -d "node_modules/axios" ] || [ ! -d "node_modules/express" ] || [ ! -d "node_modules/multer" ]; then
    echo "⚠️  Some dependencies are missing. Reinstalling..."
    echo ""
    rm -rf node_modules package-lock.json
    sudo npm install --unsafe-perm
    
    if [ $? -ne 0 ]; then
        echo ""
        echo "❌ npm install failed!"
        echo ""
        read -p "Press Enter to close..."
        exit 1
    fi
fi
echo "✅ All dependencies verified!"
echo ""

echo "🚀 Starting server on http://localhost:3000"
echo ""
echo "======================================"
echo "✅ Server is running!"
echo "======================================"
echo ""
echo "📋 What to do now:"
echo "1. Keep this window OPEN while using the stabilizer"
echo "2. Go to your browser at:"
echo "   https://confidentcinematiclab.com/video_stabilizer.html"
echo "3. Upload and stabilize your videos!"
echo ""
echo "⚠️  DO NOT CLOSE this window until you're done stabilizing"
echo ""
echo "Press Ctrl+C to stop the server when finished"
echo "======================================"
echo ""

# Start the server with PATH set
node server.js

# This line runs if the server stops
echo ""
echo "Server stopped."
read -p "Press Enter to close..."
